/*
 * xbmstreamd	XBMSP 1.0 streaming server.
 *
 * Version:	$Id: $
 *
 * Copyright:	(C)2007-2008 Miquel van Smoorenburg
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <fcntl.h>
#include <sys/uio.h>
#include <sys/wait.h>
#include <time.h>
#include <poll.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#if USE_PTHREADS
#  include <pthread.h>
#endif

#include "xbmsp.h"
#include "xbmstreamd.h"

int debuglvl = 0;

extern int h_errno;

/* Open connection */
struct conn {
	int			clientfd;
	int			id;
	struct xbmsp_packet	*client_rq;	/* Client requests */
	struct backend		*(*binit)(struct backendopts *, int id,
						char *claddr, void *clconn);
	struct backendopts	*bopts;
	char			claddr[64];
};

struct discoveropts {
	char			*servername;
	int			fd;
};

/*
 *	Got SIGCHLD, clean up zombies.
 */
void chld_handler(int sig)
{
	while (waitpid(-1, NULL, WNOHANG) > 0)
		;
}

/*
 *	Log a message.
 */
void logmsg(int severity, char *fmt, ...)
{
	va_list		ap;
	FILE		*fp;

	fp = (severity == LOG_DEBUG) ? stdout : stderr;

	va_start(ap, fmt);
	fprintf(fp, "xbmstreamd: ");
	vfprintf(fp, fmt, ap);
	fprintf(fp, "\n");
	va_end(ap);
}

void dumppacket(int id, char *label, struct xbmsp_packet *pkt)
{
	char			buf[256];

	if (debuglvl == 0)
		return;

	xbmsp_dumppacket(pkt, buf, sizeof(buf));
	logmsg(LOG_DEBUG, "%d: %s: %s", id, label, buf);
}

void fh_inc_generation(struct filehandle *fh)
{
	if (fh->generation < FH_GENERATION_MAX) {
		fh->generation++;
		if (fh->generation == FH_GENERATION_MAX)
			fh->generation = 1;
	}
}

void fh_add(struct filehandle **fl, uint32_t handle)
{
	struct filehandle	*f;

	f = malloc(sizeof(struct filehandle));
	memset(f, 0, sizeof(*f));
	f->handle = handle;
	f->generation = 1;

	f->next = *fl;
	*fl = f;
}

struct filehandle *fh_find(struct filehandle *fl, uint32_t handle)
{
	struct filehandle	*f;

	for (f = fl; f; f = f->next)
		if (f->handle == handle)
			break;

	return f;
}

/*
 *	File is closed. Remove all cached data, remove filehandle,
 *	invalidate any outstanding requests against this file.
 */
void fh_delete(struct filehandle **fh, uint32_t handle,
		void (*cleanup)(struct filehandle *))
{
	struct xbmsp_packet	*pkt, *pnext;
	struct filehandle	*f, **flast;

	/* Find filehandle */
	flast = fh;
	for (f = *fh; f; f = f->next) {
		if (f->handle == handle)
			break;
		flast = &f->next;
	}
	if (f == NULL)
		return;

	if (cleanup)
		cleanup(f);

	if (f->type == FH_TYPE_FILE && f->fd >= 0)
		close(f->fd);
	if (f->type == FH_TYPE_DIR && f->dir)
		closedir(f->dir);

	/* Remove file data */
	for (pkt = f->fileread_rp; pkt; pkt = pnext) {
		pnext = pkt->next;
		free(pkt);
	}

	/* Remove file handle */
	*flast = f->next;
	free(f);
}

void fh_deleteall(struct filehandle **fh, void (*cleanup)(struct filehandle *))
{
	while (*fh)
		fh_delete(fh, (*fh)->handle, cleanup);
}

/*
 *	Run one client connection.
 */
void *session(void *aux)
{
	struct pollfd		pfd[2];
	struct xbmsp_packet	*pkt;
	struct xbmsp_packet	*rpkt, **rplast;
	struct conn		*c = (struct conn *)aux;
	struct backend		*b;
	char			ident[128];
	int			e;
	int			numfds;
	int			quit = 0;
	int			r = 0;

	b = c->binit(c->bopts, c->id, c->claddr, c);
	if (b == NULL)
		goto session_exit;

	/* Write banner to client. */
	e = xbmsp_sendident(c->clientfd, XBMSP_VERSION " " XVERSION "\r\n", 0);
	if (e < 0) {
		logmsg(LOG_ERR, "%d: writing banner to client: %s",
			c->id, xbmsp_strerror(e));
		goto session_exit;
	}

	/* Read banner from client */
	e = xbmsp_readident(c->clientfd, ident, sizeof(ident));
	if (e < 0) {
		logmsg(LOG_ERR, "%d: reading banner from client: %s",
			c->id, xbmsp_strerror(e));
		goto session_exit;
	}

	/* Connect to the backend. */
	if (b->connect && b->connect(b) < 0)
		goto session_exit;

	logmsg(LOG_INFO, "%d: connected", c->id);

	numfds = 1;
	memset(pfd, 0, sizeof(pfd));
	pfd[0].fd = c->clientfd;
	pfd[0].events = POLLIN;
	if (b->netfd) {
		pfd[1].fd = b->netfd(b);
		pfd[1].events = POLLIN;
		numfds++;
	}

	while (!quit) {

		r = 0;

		if (poll(pfd, numfds, -1) < 0) {
			if (errno == EINTR)
				continue;
			logmsg(LOG_ERR, "%d: poll: %s", c->id, strerror(errno));
			r = -1;
			break;
		}

		if (pfd[0].revents & POLLIN) {

			/* Packet received from client */
			e = xbmsp_recvpacket(c->clientfd, &pkt);
			if (pkt) dumppacket(c->id, "<<CLNT", pkt);
			if (pkt == NULL || e < 0) {
				if (e < 0) {
					logmsg(LOG_ERR, "%d: read from client: %s",
						c->id, xbmsp_strerror(e));
					r = e;
				} else
					logmsg(LOG_INFO, "%d: EOF", c->id);
				if (pkt != NULL) free(pkt);
				break;
			}

			/* Send request packet to the backend. */
			e = b->request(b, xbmsp_clonepacket(pkt));
			if (e < 0) {
				r = -1;
				break;
			}
			/* And put it on the outstanding-request queue. */
			rplast = &c->client_rq;
			for (rpkt = c->client_rq; rpkt; rpkt = rpkt->next)
				rplast = &rpkt->next;
			*rplast = pkt;
		}

		if (pfd[1].revents & POLLIN) {

			/* Tell backend there is data */
			e = b->netio(b, pfd[1].revents);
			if (e <= 0) {
				quit = 1;
				r = e;
			}
		}

		/* See if we can send packets to the client. */
		e = 0;
		while (c->client_rq && e >= 0) {

			pkt = c->client_rq;

			/* Any server reply yet ? */
			b->reply(b, pkt->msgid, &rpkt);
			if (rpkt == NULL)
				break;

			/* Dequeue request */
			c->client_rq = pkt->next;

			e = xbmsp_sendpacket(c->clientfd, rpkt);
			dumppacket(c->id, ">>CLNT", rpkt);
			free(pkt);
			free(rpkt);
			if (e < 0) {
				logmsg(LOG_ERR, "%d: sending packet to "
					"client %s: ", c->id, xbmsp_strerror(e));
				r = e;
				quit = 1;
				break;
			}
		}
	}

	logmsg(LOG_INFO, "%d: disconnecting.", c->id);

	/* Clean up. */
session_exit:
	if (b && b->shutdown)
		b->shutdown(b);
	xbmsp_freepackets(&c->client_rq);
	close(c->clientfd);

	free(b);
	free(c);

	return NULL;
}

void *discovery(void *arg)
{
	struct xbmsp_packet	*pkt;
	struct sockaddr_storage	ss;
	struct discoveropts	*dio;
	char			*verstr, *comstr;
	char			buf[64];
	int			sslen;
	int			s, n;
	int			msgid;
	int			comlen, verlen;

	dio = (struct discoveropts *)arg;
	s = dio->fd;

	verstr = XBMSP_VERSION " " XVERSION;
	verlen = strlen(verstr);

	comstr = dio->servername;
	comlen = strlen(comstr);

	while (1) {

		/* Read packet */
		sslen = sockaddr_size(&ss);
		n = xbmsp_recvpacketfrom(s, &pkt,
					(struct sockaddr *)&ss, &sslen);
		if (n != 0) {
			if (pkt) free(pkt);
			continue;
		}

		if (pkt->type != XBMSP_PACKET_SERVER_DISCOVERY_QUERY) {
			free(pkt);
			continue;
		}

		getnameinfo((struct sockaddr *)&ss, sslen,
					buf, sizeof(buf),
					NULL, 0, NI_NUMERICHOST);
		logmsg(LOG_DEBUG, "Replying to discovery packet from %s", buf);

		msgid = pkt->msgid;
		free(pkt);

		xbmsp_buildpacket(&pkt,
				XBMSP_PACKET_SERVER_DISCOVERY_REPLY, msgid,
				0, "", 0, "",
				strlen(verstr), verstr,
				strlen(comstr), comstr);

		sslen = sizeof(ss);
		xbmsp_sendpacketto(s, pkt, (struct sockaddr *)&ss, sslen);

		free(pkt);
	}

	/*NOTREACHED*/
	return NULL;
}

static int backendopts_resolve(struct backendopts *bopts)
{
	struct addrinfo		hints, *ai;
	int			e;

        /* Resolve the remote host. */
	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AI_ADDRCONFIG;
	hints.ai_family = AF_UNSPEC;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = IPPROTO_TCP;
	if (bopts->af_family)
		hints.ai_family = bopts->af_family;

	e = getaddrinfo(bopts->host, bopts->port, &hints, &ai);
	if (e != 0) {
                logmsg(LOG_ERR, "%s:%s: %s",
			bopts->host, bopts->port, gai_strerror(e));
                return -1;
        }
	memcpy(&bopts->addr, ai->ai_addr, sizeof(bopts->addr));
	freeaddrinfo(ai);

	return 0;
}

void daemonize(void)
{
	pid_t		pid;
	int		fd;

	if ((pid = fork()) < 0) {
		logmsg(LOG_ERR, "fork: %s\n", strerror(errno));
		exit(1);
	}
	if (pid > 0)
		exit(0);
	if ((fd = open("/dev/null", O_RDWR)) >= 0) {
		dup2(fd, 0);
		dup2(fd, 1);
		dup2(fd, 2);
		if (fd > 2)
			close(fd);
	}
	setsid();
}

void usage(void)
{
	fprintf(stderr, "Usage: \n");
	fprintf(stderr, "  xbmstreamd [-c sz] [-r 4|6] -h remotehost[:port]\n");
	fprintf(stderr, "  xbmstreamd -d directory\n");
	fprintf(stderr, " commonopts: [-l 4|6] [-p port] [-n name] [-P pidfile] [-a] [-t file] [-f] [-D]\n");
	fprintf(stderr, "  -l:  local: 4 = IPv4 only, 6 = IPv6 only\n");
	fprintf(stderr, "  -r:  remote: 4 = IPv4 only, 6 = IPv6 only\n");
	fprintf(stderr, "  -a:  make server auto-discoverable on lan\n");
	fprintf(stderr, "  -c:  cache size (default: 1048576)\n");
	fprintf(stderr, "  -t:  htaccess filename\n");
	fprintf(stderr, "  -n:  system name\n");
	fprintf(stderr, "  -f:  run in the foreground\n");
	fprintf(stderr, "  -D:  print protocol debug output (-DD = more)\n");

	exit(1);
}

int main(int argc, char **argv)
{
	FILE			*fp;
	struct sigaction	sig;
	struct addrinfo		hints, *ai;
	struct sockaddr_storage	ss;
	struct conn		*c;
	struct discoveropts	dio;
	struct			backendopts bopts;
	pid_t			discovery_pid = 0;
	char			hostname[128];
	char			*remotehost, *directory, *name;
	char 			*remoteport, *p;
	char			*localport;
	socklen_t		sslen;
#if USE_PTHREADS
	pthread_t		discovery_thr, session_thr;
#else
	pid_t			pid;
#endif
	int			local_family;
	int			cachesz;
	int			connid = 1;
	int			o, e;
	int			s, val;
	int			dodisc = 0, dodaemon = 1;
	char			*pidfile = NULL;
	char			*htfile = NULL;

	localport = "1400";
	directory = NULL;
	cachesz = -1;
	name = NULL;
	local_family = AF_UNSPEC;

	remoteport = "1400";
	remotehost = NULL;
	memset(&bopts, 0, sizeof(bopts));
	bopts.af_family = AF_UNSPEC;

	while ((o = getopt(argc, argv, "l:r:DP:p:h:d:n:c:t:af")) != -1) switch(o) {
		case 'l':
			if (strcmp(optarg, "4") == 0)
				local_family = AF_INET;
			else if (strcmp(optarg, "6") == 0)
				local_family = AF_INET6;
			else
				usage();
			break;
		case 'r':
			if (strcmp(optarg, "4") == 0)
				bopts.af_family = AF_INET;
			else if (strcmp(optarg, "6") == 0)
				bopts.af_family = AF_INET6;
			else
				usage();
			break;
		case 'a':
			dodisc = 1;
			break;
		case 'f':
			dodaemon = 0;
			break;
		case 'p':
			localport = optarg;
			break;
		case 'P':
			pidfile = optarg;
			break;
		case 'h':
			remotehost = optarg;
			break;
		case 'd':
			directory = optarg;
			break;
		case 'D':
			debuglvl++;
			break;
		case 'c':
			cachesz = atoi(optarg);
			break;
		case 'n':
			name = optarg;
		case 't':
			htfile = optarg;
			break;
		default:
			usage();
	}
	if (optind < argc) usage();

	if ((!remotehost && !directory) || (remotehost && directory))
		usage();

	gethostname(hostname, sizeof(hostname));

	setlinebuf(stdout);
	setlinebuf(stderr);

	/* Set backend options */
	if (remotehost && (p = strchr(remotehost, ':')) != NULL) {
		*p++ = 0;
		remoteport = p;
	}
	bopts.name = name ? name : (remotehost ? remotehost : hostname);
	bopts.host = remotehost;
	bopts.port = remoteport;
	bopts.directory = directory;
	bopts.htaccess = htfile;
	bopts.cachesz = cachesz;
	if (bopts.host && backendopts_resolve(&bopts) < 0)
		return 1;

	/* Sockets listen on this address/port */
	memset(&hints, 0, sizeof(hints));
	hints.ai_flags = AI_PASSIVE | AI_ADDRCONFIG;
	hints.ai_family = local_family;
	hints.ai_socktype = SOCK_STREAM;
	hints.ai_protocol = IPPROTO_TCP;
	e = getaddrinfo(NULL, localport, &hints, &ai);
	if (e != 0) {
		logmsg(LOG_ERR, "getaddrinfo: %s: %s",
			localport, gai_strerror(e));
	}

	/* Open tcp listening socket */
	s = socket(ai->ai_family, SOCK_STREAM, IPPROTO_TCP);
	if (s < 0) {
		logmsg(LOG_ERR, "socket: %s", strerror(errno));
		freeaddrinfo(ai);
		return 1;
	}
	val = 1;
	setsockopt(s, SOL_SOCKET, SO_REUSEADDR, &val, sizeof(val));
#ifdef IPV6_V6ONLY
	if (ai->ai_family == AF_INET6) {
		val = (local_family == AF_INET6) ? 1 : 0;
		if (setsockopt(s, IPPROTO_IPV6, IPV6_V6ONLY,
						&val, sizeof(val)) < 0) {
			logmsg(LOG_ERR, "setsockopt IPV6_V6ONLY: %s",
							strerror(errno));
			freeaddrinfo(ai);
			close(s);
			return 1;
		}
	}
#endif
	if (bind(s, ai->ai_addr, ai->ai_addrlen) < 0) {
		logmsg(LOG_ERR, "tcp bind to local port %s: %s",
			localport, strerror(errno));
		freeaddrinfo(ai);
		close(s);
		kill(discovery_pid, SIGTERM);
		return 1;
	}

	/* Open discovery socket */
	if (dodisc) {
		dio.fd = socket(ai->ai_family, SOCK_DGRAM, IPPROTO_UDP);
		dio.servername = bopts.name;
		if (dio.fd < 0) {
			freeaddrinfo(ai);
			close(s);
			logmsg(LOG_ERR, "socket: %s", strerror(errno));
			return 1;
		}
#ifdef IPV6_V6ONLY
		if (local_family == AF_INET6) {
			val = 1;
			if (setsockopt(dio.fd, IPPROTO_IPV6, IPV6_V6ONLY,
						&val, sizeof(val)) < 0) {
				logmsg(LOG_ERR, "setsockopt IPV6_V6ONLY: %s",
							strerror(errno));
				freeaddrinfo(ai);
				close(s);
				close(dio.fd);
				return 1;
			}
		}
#endif
		if (bind(dio.fd, ai->ai_addr, ai->ai_addrlen) < 0) {
			logmsg(LOG_ERR, "udp bind to local port %s: %s",
				localport, strerror(errno));
			freeaddrinfo(ai);
			close(s);
			close(dio.fd);
			return 1;
		}
	}

	freeaddrinfo(ai);

	if (dodaemon)
		daemonize();

	if (dodisc) {
#if USE_PTHREADS
		pthread_create(&discovery_thr, NULL, discovery, &dio);
		pthread_detach(discovery_thr);
#else
		/* Discovery process */
		discovery_pid = fork();
		if (discovery_pid < 0) {
			logmsg(LOG_ERR, "fork: %s", strerror(errno));
			return 1;
		}
		if (discovery_pid == 0) {
			close(s);
			discovery(&dio);
			exit(0);
		}
		close(dio.fd);
#endif
	}

	if (pidfile && (fp = fopen(pidfile, "w")) != NULL) {
		fprintf(fp, "%d\n", (int)getpid());
		fclose(fp);
	}

	memset(&sig, 0, sizeof(sig));
	sig.sa_handler = chld_handler;
	sigaction(SIGCHLD, &sig, NULL);

	/* Wait for a connection */
	if (listen(s, 128) < 0) {
		logmsg(LOG_ERR, "listening on socket: %s", strerror(errno));
		if (discovery_pid)
			kill(discovery_pid, SIGTERM);
		if (pidfile)
			unlink(pidfile);
		return 1;
	}

	while (1) {
		c = calloc(1, sizeof(struct conn));
		sslen = sizeof(ss);
		c->clientfd = accept(s, (struct sockaddr *)&ss, &sslen);
		if (c->clientfd < 0) {
			e = errno;
			free(c);
			if (e == EINTR)
				continue;
			logmsg(LOG_ERR, "accept: %s", strerror(e));
			continue;
		}
		val = 1;
		setsockopt(c->clientfd, IPPROTO_TCP, TCP_NODELAY,
							&val, sizeof(val));
		c->id = connid++;

		getnameinfo((struct sockaddr *)&ss, sslen,
					c->claddr, sizeof(c->claddr),
					NULL, 0, NI_NUMERICHOST);
		logmsg(LOG_INFO, "%d: incoming connection from %s",
							c->id, c->claddr);

		/* Choose backend */
		if (remotehost)
			c->binit = &backend_xbmsp_init;
		else
			c->binit = &backend_file_init;
		c->bopts = &bopts;

#if USE_PTHREADS
		pthread_create(&session_thr, NULL, session, c);
		pthread_detach(session_thr);
#else
		pid = fork();
		if (pid == 0) {
			session(c);
			logmsg(LOG_INFO, "%d: exiting.", c->id);
			exit(0);
		}
		if (pid < 0)
			logmsg(LOG_ERR, "fork: %s", strerror(errno));
		close(c->clientfd);
		free(c);
#endif
	}

	if (discovery_pid)
		kill(discovery_pid, SIGTERM);

	if (pidfile)
		unlink(pidfile);

	return 0;
}

