/*
 * backend-	Backend that talks XBSMP to another server.
 * xbmsp.c
 *
 * Version:	$Id: $
 *
 * Copyright:	(C)2007-2008 Miquel van Smoorenburg <miquels@cistron.nl>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */
#include <sys/types.h>
#include <sys/time.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/uio.h>
#include <sys/wait.h>
#include <time.h>
#include <poll.h>
#include <signal.h>
#include <syslog.h>
#include <errno.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>
#if USE_PTHREADS
#  include <pthread.h>
#endif

#include "xbmsp.h"
#include "xbmstreamd.h"

extern int h_errno;

#define FH_INVALID		(FH_GENERATION_MAX+1)
#define STREAM_PACKETS		8
#define STREAM_CACHE		(1024*1024)
#define STREAM_LATENCY		32

#define DEBUG			1

#if USE_PTHREADS
#  define CONNECTION_CACHE	1
#endif

/* Open connection */
struct backend_xbmsp {
	struct backend		backend;
	int			serverfd;
	char			host[128];
	char			port[16];
	struct sockaddr_storage	addr;
	int			id;
	int			cachesz;
	unsigned int		msgid_counter;	/* server msgids */
	struct xbmsp_packet	*client_rq;	/* Client requests */
	struct xbmsp_packet	*server_rp;	/* Server replies */
	struct xbmsp_packet	*fileread_rq;	/* File contents request */
	struct filehandle	*files;		/* Open files */
};

/* Aliased parent object members */
#define clientconn	backend.clientconn

#if CONNECTION_CACHE
struct conn_cache {
	int			fd;
	unsigned int		msgid;
	char			host[128];
	time_t			when;
	int			dirty;
	struct conn_cache	*next;
};
struct conn_cache *conn_cache_list;

pthread_mutex_t conn_cache_mutex = PTHREAD_MUTEX_INITIALIZER;
pthread_cond_t conn_cache_cond = PTHREAD_COND_INITIALIZER;
pthread_t conn_cache_thr;
int conn_cache_started;

/*
 *	helper: send packet, wait for reply.
 */
static int send_recv_packet(int fd, struct xbmsp_packet *spkt,
					struct xbmsp_packet **rpkt)
{
	int			e;

	*rpkt = NULL;

	e = xbmsp_sendpacket(fd, spkt);
	if (e < 0)
		return e;

	while (1) {
		e = xbmsp_recvpacket(fd, rpkt);
		if (e < 0)
			return e;
		if ((*rpkt)->msgid == spkt->msgid)
			break;
		xbmsp_freepackets(rpkt);
		*rpkt = NULL;
	}

	return 0;
}

/*
 *	Clean one connection so it's ready to be recycled.
 */
int conn_cache_clean(struct conn_cache *c)
{
	struct xbmsp_packet	*spkt, *rpkt;
	int			e;

	xbmsp_buildpacket(&spkt, XBMSP_PACKET_SETCWD, ++c->msgid, 1, "/");
	e = send_recv_packet(c->fd, spkt, &rpkt);
	xbmsp_freepackets(&spkt);
	xbmsp_freepackets(&rpkt);
	if (e < 0)
		return e;

	xbmsp_buildpacket(&spkt, XBMSP_PACKET_CLOSE_ALL, ++c->msgid);
	e = send_recv_packet(c->fd, spkt, &rpkt);
	xbmsp_freepackets(&spkt);
	xbmsp_freepackets(&rpkt);
	if (e < 0)
		return e;

	return 0;
}
/*
 *	Connection cache thread.
 *
 *	Cleans up connections to be recycled by closing all
 *	filedescriptors and setting the working directory to "/".
 *
 *	Also removes connections older than 10 seconds.
 */
void *conn_cache_thread(void *aux)
{
	struct conn_cache	*c, *next, **last;
	struct timespec		timeout;
	struct timeval		tvnow;
	time_t			now;
	int			e;

	pthread_mutex_lock(&conn_cache_mutex);

	while (1) {

		time(&now);
		last = &conn_cache_list;

		for (c = conn_cache_list; c; c = next) {
			next = c->next;

			if (c->when < now - 10) {
				/* old fd, remove */
				*last = c->next;
				close(c->fd);
				free(c);
			} else if (c->dirty) {
				/* need to clean fd */
				pthread_mutex_unlock(&conn_cache_mutex);
				e = conn_cache_clean(c);
				pthread_mutex_lock(&conn_cache_mutex);
				if (e < 0) {
					/* error - remove */
					*last = c->next;
					close(c->fd);
					free(c);
				}
				c->dirty = 0;
				/* restart loop */
				last = &conn_cache_list;
				next = conn_cache_list;
			} else {
				last = &c->next;
			}
		}

		gettimeofday(&tvnow, NULL);
		timeout.tv_sec = tvnow.tv_sec + 2;
		timeout.tv_nsec = tvnow.tv_usec * 1000;
		pthread_cond_timedwait(&conn_cache_cond,
					&conn_cache_mutex, &timeout);
	}

	/*NOTREACHED*/
	return NULL;
}

/*
 *	Add a connection to the cache of connections.
 *	Keeps a maximum of 4 connections per host cached.
 */
void conn_cache_add(int fd, char *host, unsigned int msgid)
{
	struct conn_cache	*c;
	int			n = 0;
	int			dostart = 0;

	pthread_mutex_lock(&conn_cache_mutex);

	if (!conn_cache_started) {
		dostart = 1;
		conn_cache_started = 1;
	}

	for (c = conn_cache_list; c; c = c->next)
		n++;
	if (n >= 4) {
		pthread_mutex_unlock(&conn_cache_mutex);
		close(fd);
		return;
	}

	c = calloc(1, sizeof(struct conn_cache));
	strlcpy(c->host, host, sizeof(c->host));
	c->fd = fd;
	c->when = time(NULL);
	c->msgid = msgid;
	c->dirty = 1;

	c->next = conn_cache_list;
	conn_cache_list = c;
	pthread_cond_signal(&conn_cache_cond);
	pthread_mutex_unlock(&conn_cache_mutex);

	if (dostart)
		pthread_create(&conn_cache_thr, NULL, &conn_cache_thread, NULL);
}

/*
 *	Recycle an open connection from the cache.
 */
int conn_cache_get(char *host)
{
	struct conn_cache	*c, **last;
	int			fd = -1;

	pthread_mutex_lock(&conn_cache_mutex);
	last = &conn_cache_list;
	for (c = conn_cache_list; c; c = c->next) {
		if (!c->dirty && strcmp(c->host, host) == 0) {
			*last = c->next;
			fd = c->fd;
			free(c);
			break;
		}
		last = &c->next;
	}
	pthread_mutex_unlock(&conn_cache_mutex);

	return fd;
}

#endif

/*
 *	Count cachesize , including outstanding fileread commands.
 */
static int file_cachesize(struct backend_xbmsp *b,
			struct filehandle *fh, int *cnt)
{
	struct xbmsp_packet	*pkt;
	int			sz1 = 0, sz2 = 0;

	/* Outstanding requests */
	for (pkt = b->fileread_rq; pkt; pkt = pkt->next) {
		if (pkt->type == XBMSP_PACKET_FILE_READ &&
		    pkt->attrs[0].val32 == fh->handle) {
			sz1 += pkt->attrs[1].val32;
			if (cnt) (*cnt)++;
		}
	}

	/* Existing cache */
	for (pkt = fh->fileread_rp; pkt; pkt = pkt->next) {
		if (pkt->generation == fh->generation)
			sz2 += pkt->attrs[0].length;
	}

	/*printf("outstanding reqs: %d. cachesize: %d\n", *cnt, sz2);*/

	return sz1 + sz2;
}


/*
 *	Return network file descriptor.
 */
static int bnetfd(struct backend *be)
{
	struct backend_xbmsp		*b = (struct backend_xbmsp *)be;

	return b->serverfd;
}

/*
 *	There is data available from the network.
 *	Receive packet from server and handle it.
 */
static int bnetio(struct backend *be, int mode)
{
	struct xbmsp_packet	*pkt;
	struct xbmsp_packet	*rqpkt, **rqlast;
	struct xbmsp_packet	*rpkt, **rlast;
	struct backend_xbmsp	*b = (struct backend_xbmsp *)be;
	struct filehandle	*fh;
	int			handled = 0;
	int			e;

	e = xbmsp_recvpacket(b->serverfd, &pkt);
	if (pkt) dumppacket(b->id, "<<SRVR", pkt);
	if (e < 0 || pkt == NULL) {
		if (e < 0)
			logmsg(LOG_ERR, "%d: read from server: %s",
				b->id, xbmsp_strerror(e));
		if (pkt != NULL) free(pkt);
		close(b->serverfd);
		return e;
	}

	/*
	 *	Find matching client request.
	 */
	if (pkt->type == XBMSP_PACKET_FILE_CONTENTS)
		rqlast = &b->fileread_rq;
	else
		rqlast = &b->client_rq;
	for (rqpkt = *rqlast; rqpkt; rqpkt = rqpkt->next) {
		if (rqpkt->msgid == pkt->msgid)
			break;
		rqlast = &rqpkt->next;
	}
	if (rqpkt == NULL) {
		logmsg(LOG_ERR, "%d: reply packet: no matching request", b->id);
		free(pkt);
		return -1;
	}

	switch (pkt->type) {

	case XBMSP_PACKET_HANDLE:
		/* File was opened. Add to open-file-list. */
		fh = fh_find(b->files, pkt->attrs[0].val32);
		if (fh) {
			syslog(LOG_ERR, "%d: File Open returned in-use handle",
								b->id);
			fh_inc_generation(fh);
		} else {
			fh_add(&b->files, pkt->attrs[0].val32);
		}
		break;

	case XBMSP_PACKET_FILE_CONTENTS:

		handled = 1;

		/* Remove request from queue */
		*rqlast = rqpkt->next;
		pkt->generation = rqpkt->generation;
		fh = fh_find(b->files, rqpkt->attrs[0].val32);
		free(rqpkt);

		/* Do we know this file ? */
		if (fh == NULL) {
			logmsg(LOG_ERR, "%d: got data from server for "
					"unknown/closed file", b->id);
			free(pkt);
			break;
		}

		/* And add reply to the end of the file data queue */
		rlast = &fh->fileread_rp;
		for (rpkt = fh->fileread_rp; rpkt; rpkt = rpkt->next)
			rlast = &rpkt->next;
		*rlast = pkt;

		break;

	}

	if (handled == 0) {
		/* Add reply to the end of the queue */
		rlast = &b->server_rp;
		for (rpkt = b->server_rp; rpkt; rpkt = rpkt->next)
			rlast = &rpkt->next;
		*rlast = pkt;
	}

	return 1;
}

/*
 *	Filehandle cleanup functie.
 */
static void bcleanup(struct filehandle *fp)
{
	/*UNUSED*/
}

/*
 *	Helper for the SEEK command - if relative, we need to
 *	account for the outstanding requests.
 */
static int do_seek(struct backend_xbmsp *b, struct xbmsp_packet *pkt,
			struct filehandle *fh)
{
	struct xbmsp_packet		*dpkt;
	int				sz;
	int				outstanding;
	uint32_t			val32, val32_2;

	/* Not needed for absolute seeks. */
	if (pkt->attrs[1].val32 == 0 || pkt->attrs[1].val32 == 1)
		return 0;

	/* Drain outstanding requests. */
	outstanding = 1;
	while (outstanding) {
		outstanding = 0;
		for (dpkt = b->fileread_rq; dpkt; dpkt = dpkt->next) {
			if (dpkt->attrs[0].val32 == fh->handle &&
			    dpkt->generation == fh->generation) {
				outstanding = 1;
				break;
			}
		}
		if (outstanding)
			bnetio((struct backend *)b, POLLIN);
	}

	/* Find requests, count total size. */
	sz = 0;
	for (dpkt = fh->fileread_rp; dpkt; dpkt = dpkt->next) {
		if (dpkt->generation == fh->generation)
			sz += dpkt->attrs[0].length;
	}

	/* Adjust for sz */
	switch (pkt->attrs[1].val32) {
		case 2: /* Forward from current pos */
			if (sz > pkt->attrs[2].val64) {
				/* Already past it, need to go back */
				pkt->attrs[1].val32 = 3;
				pkt->attrs[2].val64 = sz - pkt->attrs[2].val64;
			} else {
				pkt->attrs[2].val64 -= sz;
			}
			break;
		case 3: /* Backwards from current pos */
			pkt->attrs[2].val64 += sz;
			break;
	}

	/* This is gross. XXX FIXME */
	pkt->data[4] = pkt->attrs[1].val32;
	val32 = htonl(pkt->attrs[2].val64 >> 32);
	val32_2 = htonl(pkt->attrs[2].val64 & 0xFFFFFFFF);
	memcpy(pkt->data + 5, &val32, 4);
	memcpy(pkt->data + 9, &val32_2, 4);

	return sz;
}

/*
 *	Request from the client.
 */
static int brequest1(struct backend *be, struct xbmsp_packet *pkt, int internal)
{
	struct backend_xbmsp	*b = (struct backend_xbmsp *)be;
	struct filehandle	*fh;
	struct xbmsp_packet	*rpkt, **rlast;
	int			reqs;
	int			e;

	b->msgid_counter++;
	if (b->msgid_counter > 999999999)
		b->msgid_counter = 1;

	pkt->client_msgid = pkt->msgid;
	pkt->msgid = b->msgid_counter;
	
	switch (pkt->type) {
		case XBMSP_PACKET_FILE_READ:

			fh = fh_find(b->files, pkt->attrs[0].val32);
			if (fh)
				pkt->generation = fh->generation;
			else
				logmsg(LOG_ERR, "File Read: cannot find "
					"open file w/ file handle %d\n",
					pkt->attrs[0].val32);

			/* Insert clone at the end of the fileread requests. */
			rlast = &b->fileread_rq;
			for (rpkt = b->fileread_rq; rpkt; rpkt = rpkt->next)
				rlast = &rpkt->next;
			*rlast = xbmsp_clonepacket(pkt);

			break;

		case XBMSP_PACKET_FILE_SEEK:
			/* First drain the outstanding requests */
			fh = fh_find(b->files, pkt->attrs[0].val32);
			if (fh) {
				do_seek(b, pkt, fh);
				fh_inc_generation(fh);
				fh->streaming = 0;
			}
			break;
		case XBMSP_PACKET_CLOSE:
			fh_delete(&b->files, pkt->attrs[0].val32, bcleanup);
			break;
		case XBMSP_PACKET_CLOSE_ALL:
			fh_deleteall(&b->files, bcleanup);
			break;
	}

	/* Remember request packet */
	if (!internal) {
		rlast = &b->client_rq;
		for (rpkt = b->client_rq; rpkt; rpkt = rpkt->next)
			rlast = &rpkt->next;
		*rlast = pkt;
	}

	/* And submit request to the server */
	e = xbmsp_sendpacket(b->serverfd, pkt);
	dumppacket(b->id, ">>SRVR", pkt);
	if (e < 0)
		logmsg(LOG_ERR, "%d: writing to server: %s",
			b->id, xbmsp_strerror(e));

	/* Maybe ask for some more packets */
	if (pkt->type == XBMSP_PACKET_FILE_READ && fh &&
	    fh->streaming++ >= STREAM_PACKETS &&
	    file_cachesize(b, fh, &reqs) < b->cachesz &&
	    reqs < STREAM_LATENCY) {
		struct xbmsp_packet *cpkt = xbmsp_clonepacket(pkt);
		cpkt->msgid = 0;
		brequest1(be, cpkt, 1);
	}

	return e;
}

static int brequest(struct backend *be, struct xbmsp_packet *pkt)
{
	return brequest1(be, pkt, 0);
}

/*
 *	See if there is a reply to a certain request yet.
 *	there is a reply from the server - if so return reply packet.
 */
static int breply(struct backend *be, uint32_t msgid,
					struct xbmsp_packet **rppkt)
{
	struct backend_xbmsp		*b = (struct backend_xbmsp *)be;
	struct xbmsp_packet		*pkt, *tpkt;
	struct xbmsp_packet		*rpkt, **rlast;
	struct xbmsp_packet		*dpkt, **dlast, *dnext;
	struct filehandle		*fh;

	*rppkt = NULL;

	for (pkt = b->client_rq; pkt; pkt = pkt->next)
		if (pkt->client_msgid == msgid)
			break;
	if (pkt == NULL)
		return 0;

	if (pkt->type != XBMSP_PACKET_FILE_READ) {

		/* Non-file-content packet. Find reply. */
		rlast = &b->server_rp;
		for (rpkt = b->server_rp; rpkt; rpkt = rpkt->next) {
			if (rpkt->msgid == pkt->msgid)
				break;
			rlast = &rpkt->next;
		}
		if (rpkt == NULL)
			return 0;

		rpkt->msgid = pkt->client_msgid;
		b->client_rq = pkt->next;
		free(pkt);
		*rlast = rpkt->next;

		*rppkt = rpkt;

		return 0;
	}

	/*
	 *	The client wants to read data. See if we can satisfy
	 *	the request. Partially is OK.
	 */
	fh = fh_find(b->files, pkt->attrs[0].val32);

	if (fh == NULL) {

send_empty_packet:
		/* Invalid file handle. Send empty data. */
		xbmsp_buildpacket(&rpkt, XBMSP_PACKET_FILE_CONTENTS,
			pkt->client_msgid, 0, "");

		b->client_rq = pkt->next;
		free(pkt);

		*rppkt = rpkt;

		return 0;
	}

	/*
	 *	See if we have data pending.
	 *	Where we're at it, delete data from older generations.
	 */
	dlast = &fh->fileread_rp;
	for (dpkt = fh->fileread_rp; dpkt; dpkt = dnext) {
		dnext = dpkt->next;
		if (dpkt->generation < pkt->generation ||
		    dpkt->generation > pkt->generation + FH_GENERATION_MAX/2) {
			*dlast = dnext;
			free(dpkt);
			continue;
		}
		if (dpkt->generation == pkt->generation)
			break;
		dlast = &dpkt->next;
	}

	if (dpkt == NULL) {
		/* No data. Are there outstanding requests ? */
		for (dpkt = b->fileread_rq; dpkt; dpkt = dpkt->next) {
			if (dpkt->attrs[0].val32 == pkt->attrs[0].val32 &&
			    dpkt->generation == pkt->generation)
				/* Yes, so just be patient */
				return 0;
		}

		/*
		 *	XXX FIXME: No data. Should not happen.
		 *	We should request more data, but for now,
		 *	just send an empty packet.
		 */
		logmsg(LOG_ERR, "%d: File Read %d: no data, no pending "
				"requests", b->id, pkt->attrs[0].val32);
		goto send_empty_packet;
	}

	/* We have data, so send it to the client */

	/* Read request smaller than available data ? */
	if (pkt->attrs[1].val32 < dpkt->attrs[0].length) {
		/*
		 *	Cannot re-use datapacket as reply
		 *	packet, so build fresh reply packet,
		 *	and shrink datapacket.
		 */
		xbmsp_buildpacket(&rpkt, XBMSP_PACKET_FILE_CONTENTS,
				pkt->client_msgid, pkt->attrs[1].length,
				dpkt->attrs[0].string);
		xbmsp_buildpacket(&tpkt, XBMSP_PACKET_FILE_CONTENTS,
				dpkt->msgid,
				dpkt->attrs[0].length - pkt->attrs[0].length,
				dpkt->attrs[0].string + pkt->attrs[0].length);
		tpkt->generation = dpkt->generation;
		*dlast = tpkt;
		tpkt->next = dpkt->next;
		free(dpkt);
	} else {
		/*
		 *	Request >= data available, so just use as-is.
		 */
		*dlast = dpkt->next;
		rpkt = dpkt;
		rpkt->msgid = pkt->client_msgid;
	}

	/* Send packet to client */
	b->client_rq = pkt->next;
	free(pkt);

	*rppkt = rpkt;

	return 0;
}

static int bconnect(struct backend *be)
{
	struct backend_xbmsp	*b = (struct backend_xbmsp *)be;
        char                    ident[128];
        int                     s, e;

#if CONNECTION_CACHE
	s = conn_cache_get(b->host);
	if (s >= 0) {
        	logmsg(LOG_INFO, "%d: re-using connection to %s",
							b->id, b->host);
		b->serverfd = s;
		return 0;
	}
#endif
        s = socket(b->addr.ss_family, SOCK_STREAM, IPPROTO_TCP);
        if (s < 0) {
                logmsg(LOG_ERR, "socket: %s", strerror(errno));
                return -1;
        }
        logmsg(LOG_INFO, "%d: connecting to %s", b->id, b->host);
        if (connect(s, (struct sockaddr *)&b->addr,
	    sockaddr_size(&b->addr)) < 0) {
                logmsg(LOG_ERR, "connect %s: %s", b->host, strerror(errno));
                return -1;
        }
        b->serverfd = s;

        /* Read banner from server */
        e = xbmsp_readident(b->serverfd, ident, sizeof(ident));
        if (e < 0) {
                logmsg(LOG_ERR, "reading banner from server: %s",
                        xbmsp_strerror(e));
                return -1;
        }

        /* Write banner to client. */
        e = xbmsp_sendident(b->serverfd, XBMSP_VERSION " " XVERSION "\r\n", 0);
        if (e < 0) {
                logmsg(LOG_ERR, "writing banner to server: %s",
                        xbmsp_strerror(e));
                return -1;
        }

	return 0;
}

/*
 *	Destructor.
 */
static int bshutdown(struct backend *be)
{
	struct backend_xbmsp		*b = (struct backend_xbmsp *)be;

#if CONNECTION_CACHE
	conn_cache_add(b->serverfd, b->host, b->msgid_counter);
#else
	close(b->serverfd);
#endif
	xbmsp_freepackets(&b->client_rq);
	xbmsp_freepackets(&b->server_rp);
	xbmsp_freepackets(&b->fileread_rq);
	fh_deleteall(&b->files, NULL);

	return 0;
}

struct backend backend_xbmsp = {
	.name		= "xbmsp-proxy",
	.connect	= bconnect,
	.request	= brequest,
	.reply		= breply,
	.netfd		= bnetfd,
	.netio		= bnetio,
	.shutdown	= bshutdown,
};

/*
 *	Constructor.
 */
struct backend *backend_xbmsp_init(struct backendopts *bopts, int id,
						char *claddr, void *clconn)
{
	struct backend_xbmsp	*b;

	b = calloc(1, sizeof(struct backend_xbmsp));
	memcpy(b, &backend_xbmsp, sizeof(struct backend));
	b->id = id;
	strlcpy(b->host, bopts->host, sizeof(b->host));
	strlcpy(b->port, bopts->port, sizeof(b->port));
	b->clientconn = clconn;
	b->addr = bopts->addr;
	b->cachesz = bopts->cachesz >= 0 ? bopts->cachesz : STREAM_CACHE;

	return (struct backend *)b;
}

