/*
 * backend-	Backend that just presents the filesystem.
 * xbmsp.c
 *
 * Version:	$Id: $
 *
 * Copyright:	(C)2007 Miquel van Smoorenburg <miquels@cistron.nl>
 *
 *		This program is free software; you can redistribute it and/or
 *		modify it under the terms of the GNU General Public License
 *		as published by the Free Software Foundation; either version
 *		2 of the License, or (at your option) any later version.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdarg.h>
#include <unistd.h>
#include <stdlib.h>
#include <string.h>
#include <sys/stat.h>
#include <time.h>
#include <syslog.h>
#include <errno.h>
#include <fcntl.h>
#include <arpa/inet.h>
#include <netdb.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netinet/tcp.h>

#include "xbmsp.h"
#include "xbmstreamd.h"

#define MAXFNSZ			256

/* Open connection */
struct backend_file {
	struct backend		backend;
	char			fullcwd[MAXFNSZ];
	char			*cwd;
	int			id;
	int			handle_counter;
	struct filehandle	*files;		/* Open files */
	struct xbmsp_packet	*server_rp;
};

/* Aliased parent object members */
#define clientconn	backend.clientconn

struct strbuf {
	char			*buf;
	int			len;
	int			sz;
};
typedef struct strbuf strbuf;

void strbuf_append(strbuf *buf, char *str)
{
	int			l;
	int			rs = 0;

	l = strlen(str);
	while (buf->len + l >= buf->sz) {
		buf->sz += 128;
		rs = 1;
	}
	if (rs)
		buf->buf = realloc(buf->buf, buf->sz);
	memcpy(buf->buf + buf->len, str, l + 1);
	buf->len += l;
}

void strbuf_free(strbuf *buf)
{
	free(buf->buf);
	memset(buf, 0, sizeof(strbuf));
}

#define INIT_STRBUF { NULL, 0, 0 }

static void escape_xml_and_weird_chars(char *fn)
{
	while (*fn) {
		if (*fn < 32 || strchr("\\<>/", *fn))
			*fn = '.';
		fn++;
	}
}

char *xbmsp_xmlstat(char *path, char *filename, struct stat *st)
{
	strbuf		sb = INIT_STRBUF;
	char		*fn;
	char		tmp[32];

	fn = strdup(filename);
	escape_xml_and_weird_chars(fn);

	strbuf_append(&sb, "<DIRECTORYITEM>");
	strbuf_append(&sb, "<NAME>");
	strbuf_append(&sb, fn);
	strbuf_append(&sb, "</NAME>");
	strbuf_append(&sb, "<ATTRIB>");
	strbuf_append(&sb, S_ISDIR(st->st_mode) ? "directory" : "file");
	strbuf_append(&sb, "</ATTRIB>");
	strbuf_append(&sb, "<SIZE>");
	snprintf(tmp, sizeof(tmp), "%llu", (unsigned long long)st->st_size);
	strbuf_append(&sb, tmp);
	strbuf_append(&sb, "</SIZE>");
	strbuf_append(&sb, "<TIME>");
	strbuf_append(&sb, "<ACCESS>");
	snprintf(tmp, sizeof(tmp), "%llu", (unsigned long long)st->st_atime);
	strbuf_append(&sb, tmp);
	strbuf_append(&sb, "</ACCESS>");
	strbuf_append(&sb, "<MODIFICATION>");
	snprintf(tmp, sizeof(tmp), "%llu", (unsigned long long)st->st_mtime);
	strbuf_append(&sb, tmp);
	strbuf_append(&sb, "</MODIFICATION>");
	strbuf_append(&sb, "<CHANGE>");
	snprintf(tmp, sizeof(tmp), "%llu", (unsigned long long)st->st_ctime);
	strbuf_append(&sb, tmp);
	strbuf_append(&sb, "</CHANGE>");
	strbuf_append(&sb, "</TIME>");
	strbuf_append(&sb, "</DIRECTORYITEM>");

	free(fn);

	return sb.buf;
}

static int bupcwd(struct backend_file *b, int levels)
{
	char		*p;

	if (b->cwd[0] != '/')
		return 0;

	while (levels-- > 0) {
		/* Find last char of cwd */
		for (p = b->cwd; *p; p++)
			;
		p--;
		while (*p == '/' && p > b->cwd)
			*p-- = 0;
		while (*p != '/' && p > b->cwd)
			*p-- = 0;
	}

	return 0;
}

static int bsetcwd(struct backend_file *b, char *str, int sz)
{
	struct stat	st;
	char		dir[MAXFNSZ];
	char		*ptr;

	/* Chdir to the root always works */
	snprintf(dir, sizeof(dir), "%.*s", sz, str);
	if (strcmp(dir, "/") == 0) {
		if (b->cwd[0])
			b->cwd[1] = 0;
		return 0;
	}

	/* Chdir to current dir always works */
	if (strcmp(dir, ".") == 0 || dir[0] == 0)
		return 0;

	/* For ".." we use bupcwd */
	if (strcmp(dir, "..") == 0)
		return bupcwd(b, 1);

	/* Empty cwd means invalid root */
	if (b->cwd[0] == 0)
		return XBMSP_ERROR_NO_SUCH_FILE;

	/* New dirname may not contain / or \\ */
	if (strchr(dir, '/') || strchr(dir, '\\'))
		return XBMSP_ERROR_NO_SUCH_FILE;

	if (strlen(b->fullcwd) + strlen(dir) + 8 > sizeof(b->fullcwd)) {
		logmsg(LOG_ERR, "%d: setcwd: directory name too long");
		return XBMSP_ERROR_NO_SUCH_FILE;
	}

	ptr = b->fullcwd + strlen(b->fullcwd);
	strlcat(b->fullcwd, dir, sizeof(b->fullcwd));
	strlcat(b->fullcwd, "/", sizeof(b->fullcwd));
	if (stat(b->fullcwd, &st) < 0 || !S_ISDIR(st.st_mode)) {
		*ptr = 0;
		return XBMSP_ERROR_NO_SUCH_FILE;
	}

	return 0;
}

static int bopendir(struct backend_file *b, uint32_t *handle)
{
	struct filehandle	*fh;
	
	fh = calloc(1, sizeof(struct filehandle));
	fh->type = FH_TYPE_DIR;
	fh->dir = opendir(b->fullcwd);
	b->handle_counter++;
	if (b->handle_counter > 999999999)
		b->handle_counter = 1;
	fh->handle = b->handle_counter;
	*handle = fh->handle;

	fh->next = b->files;
	b->files = fh;

	return 0;
}

static int breaddir(struct backend_file *b, int handle,
					uint32_t *fnsz, char **fn,
					uint32_t *sz, char **str)
{
	struct dirent		*d;
	struct stat		st;
	struct filehandle	*fh;
	char			path[MAXFNSZ];
	int			l;

	fh = fh_find(b->files, handle);
	if (fh == NULL || fh->type != FH_TYPE_DIR || fh->dir == NULL) {
		*fn = NULL;
		*fnsz = 0;
		*str = NULL;
		*sz = 0;
		return 0;
	}

	strlcpy(path, b->fullcwd, sizeof(path));
	l = strlen(path);

	while (1) {
		d = readdir(fh->dir);
		if (d == NULL)
			break;
		if (d->d_name[0] == '.')
			continue;
		path[l] = 0;
		strlcat(path, d->d_name, sizeof(path));
		if (stat(path, &st) == 0)
			break;
	}

	if (d == NULL) {
		fh_delete(&b->files, handle, NULL);
		*fn = NULL;
		*fnsz = 0;
		*str = NULL;
		*sz = 0;
		return 0;
	}

	*fn = strdup(d->d_name);
	*fnsz = strlen(d->d_name);
	*str = xbmsp_xmlstat(b->cwd, d->d_name, &st);
	*sz = strlen(*str);

	return 0;
}

static int bfileinfo(struct backend_file *b, char *str, int sz,
					uint32_t *rsz, char **rstr)
{
	char			fn[MAXFNSZ];
	char			name[MAXFNSZ];
	struct stat		st;

	snprintf(fn, sizeof(fn), "%s%.*s", b->fullcwd, sz, str);
	if (stat(fn, &st) < 0)
		return XBMSP_ERROR_NO_SUCH_FILE;

	snprintf(name, sizeof(name), "%.*s", sz, str);

	*rstr = xbmsp_xmlstat(b->cwd, name, &st);
	*rsz = strlen(*rstr);

	return 0;
}

static int bfileopen(struct backend_file *b, char *str, int sz,
						uint32_t *handle)
{
	struct filehandle	*fh;
	char			fn[MAXFNSZ];
	int			fd;

	snprintf(fn, sizeof(fn), "%s%.*s", b->fullcwd, sz, str);
	if ((fd = open(fn, O_RDONLY)) < 0)
		return XBMSP_ERROR_NO_SUCH_FILE;

	fh = calloc(1, sizeof(struct filehandle));
	fh->type = FH_TYPE_FILE;
	fh->fd = fd;
	b->handle_counter++;
	if (b->handle_counter > 999999999)
		b->handle_counter = 1;
	fh->handle = b->handle_counter;
	*handle = fh->handle;

	fh->next = b->files;
	b->files = fh;

	return 0;
}

static int bfileread(struct backend_file *b, int handle, int length,
						uint32_t *sz, char **str)
{
	struct filehandle	*fh;
	int			n;

	fh = fh_find(b->files, handle);
	if (fh == NULL || fh->type != FH_TYPE_FILE)
		return XBMSP_ERROR_INVALID_HANDLE;

	*str = malloc(length);
	n = read(fh->fd, *str, length);
	if (n < 0) {
		free(*str);
		return XBMSP_ERROR_FAILURE;
	}

	*sz = n;

	return 0;
}

static int bfileseek(struct backend_file *b, int handle, int seektype,
						uint64_t offset)
{
	struct filehandle	*fh;
	int			whence = SEEK_SET;
	off_t			off = offset;

	fh = fh_find(b->files, handle);
	if (fh == NULL || fh->type != FH_TYPE_FILE)
		return XBMSP_ERROR_INVALID_HANDLE;

	switch (seektype) {
		case 0:
		default:
			break;
		case 1:
			whence = SEEK_END;
			off = -offset;
			break;
		case 2:
			whence = SEEK_CUR;
			break;
		case 3:
			whence = SEEK_CUR;
			off = -offset;
			break;
	}

	lseek(fh->fd, off, whence);

	return 0;
}

static int bclose(struct backend_file *b, int handle)
{
	struct filehandle	*fh;

	fh = fh_find(b->files, handle);
	if (fh == NULL || fh->type != FH_TYPE_FILE)
		return XBMSP_ERROR_INVALID_HANDLE;

	fh_delete(&b->files, fh->handle, NULL);
	return 0;
}

static int bcloseall(struct backend_file *b)
{
	fh_deleteall(&b->files, NULL);
	return 0;
}

/*
 *	Request from the client.
 */
static int brequest(struct backend *be, struct xbmsp_packet *pkt)
{
	struct backend_file	*b = (struct backend_file *)be;
	struct xbmsp_packet	*rpkt;
	struct xbmsp_packet	*rppkt, **rplast;
	int			e;
	uint32_t		val32, val32_2;
	char			*s, *fn;

	rpkt = NULL;
	e = 0;

	switch (pkt->type) {
		case XBMSP_PACKET_NULL:
			xbmsp_buildpacket(&rpkt, XBMSP_PACKET_OK,
								pkt->msgid);
			break;
		case XBMSP_PACKET_SETCWD:
			e = bsetcwd(b, pkt->attrs[0].string,
						pkt->attrs[0].length);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_OK, pkt->msgid);
			break;
		case XBMSP_PACKET_UPCWD:
			e = bupcwd(b, pkt->attrs[0].val32);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_OK, pkt->msgid);
			break;
		case XBMSP_PACKET_FILELIST_OPEN:
			e = bopendir(b, &val32);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_HANDLE, pkt->msgid, val32);
			break;
		case XBMSP_PACKET_FILELIST_READ:
			e = breaddir(b, pkt->attrs[0].val32,
				&val32, &fn, &val32_2, &s);
			if (e == 0) {
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_FILE_DATA, pkt->msgid,
					val32, fn, val32_2, s);
				if (fn) free(fn);
				if (s)  free(s);
			}
			break;
		case XBMSP_PACKET_FILE_INFO:
			e = bfileinfo(b, pkt->attrs[0].string,
					pkt->attrs[0].length, &val32, &s);
			if (e == 0) {
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_FILE_DATA, pkt->msgid,
					pkt->attrs[0].length,
					pkt->attrs[0].string,
					val32, s);
				if (s) free(s);
			}
			break;
		case XBMSP_PACKET_FILE_OPEN:
			e = bfileopen(b, pkt->attrs[0].string,
						pkt->attrs[0].length, &val32);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_HANDLE, pkt->msgid, val32);
			break;
		case XBMSP_PACKET_FILE_READ:
			e = bfileread(b, pkt->attrs[0].val32,
					pkt->attrs[1].val32, &val32, &s);
			if (e == 0) {
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_FILE_CONTENTS, pkt->msgid,
					val32, s);
				free(s);
			}
			break;
		case XBMSP_PACKET_FILE_SEEK:
			e = bfileseek(b, pkt->attrs[0].val32,
					pkt->attrs[1].val32,
					pkt->attrs[2].val64);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_OK, pkt->msgid);
			break;
		case XBMSP_PACKET_CLOSE:
			e = bclose(b, pkt->attrs[0].val32);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_OK, pkt->msgid);
			break;
		case XBMSP_PACKET_CLOSE_ALL:
			e = bcloseall(b);
			if (e == 0)
				xbmsp_buildpacket(&rpkt,
					XBMSP_PACKET_OK, pkt->msgid);
			break;
		case XBMSP_PACKET_SET_CONFIGURATION_OPTION:
		case XBMSP_PACKET_AUTHENTICATION_INIT:
		case XBMSP_PACKET_AUTHENTICATE:
		default:
			e = XBMSP_ERROR_UNSUPPORTED;
			break;
	}

	if (rpkt == NULL)
		xbmsp_buildpacket(&rpkt,
			XBMSP_PACKET_ERROR, pkt->msgid, e, 0, "");

	rplast = &b->server_rp;
	for(rppkt = b->server_rp; rppkt; rppkt = rppkt->next)
		rplast = &rppkt->next;
	*rplast = rpkt;

	/* We don't keep requests around. */
	free(pkt);

	return 0;
}

/*
 *	See if there is a reply to a certain request yet.
 *	there is a reply from the server - if so return reply packet.
 */
static int breply(struct backend *be, uint32_t msgid,
					struct xbmsp_packet **rppkt)
{
	struct backend_file		*b = (struct backend_file *)be;
	struct xbmsp_packet		*rpkt, **rlast;

	*rppkt = NULL;

	rlast = &b->server_rp;
	for (rpkt = b->server_rp; rpkt; rpkt = rpkt->next) {
		if (rpkt->msgid == msgid)
			break;
		rlast = &rpkt->next;
	}
	if (rpkt == NULL)
			return 0;

	*rlast = rpkt->next;
	*rppkt = rpkt;

	return 0;
}

/*
 *	Destructor.
 */
static int bshutdown(struct backend *be)
{
	struct backend_file		*b = (struct backend_file *)be;

	xbmsp_freepackets(&b->server_rp);
	fh_deleteall(&b->files, NULL);

	return 0;
}

struct backend backend_file = {
	.name		= "xbmsp-proxy",
	.request	= brequest,
	.reply		= breply,
	.shutdown	= bshutdown,
};

int htaccess(struct backendopts *bopts, char *claddr)
{
	FILE		*fp;
	char		fn[128];
	char		w1[32], w2[32], w3[32];
	char		*htfile;
	int		r = 0;

	htfile = bopts->htaccess;
	if (htfile == NULL) {
		snprintf(fn, sizeof(fn), "%s/.htaccess", bopts->directory);
		htfile = fn;
	}
	if ((fp = fopen(htfile, "r")) == NULL)
		return 1;

	while (fgets(fn, sizeof(fn), fp) != NULL) {
		if (sscanf(fn, " %31s %31s %31s", w1, w2, w3) != 3)
			continue;
		if (strcasecmp(w1, "allow") == 0 &&
		    strcasecmp(w2, "from") == 0 &&
		    strcasecmp(w3, claddr) == 0) {
			r = 1;
			break;
		}
	}
	fclose(fp);

	return r;
}

/*
 *	Constructor.
 */
struct backend *backend_file_init(struct backendopts *bopts, int id,
					char *claddr, void *clconn)
{
	struct backend_file	*b;

	if (htaccess(bopts, claddr) <= 0) {
		logmsg(LOG_INFO, "%d: access denied.\n");
		return NULL;
	}

	b = calloc(1, sizeof(struct backend_file));
	memcpy(b, &backend_file, sizeof(struct backend));
	b->id = id;
	if (strlen(bopts->directory) < sizeof(b->fullcwd) - 32) {
		strlcpy(b->fullcwd, bopts->directory, sizeof(b->fullcwd));
		if (strcmp(bopts->directory, "/") != 0) {
			b->cwd = b->fullcwd + strlen(b->fullcwd);
			strlcat(b->fullcwd, "/", sizeof(b->fullcwd));
		} else {
			b->cwd = b->fullcwd;
		}
	}
	b->clientconn = clconn;

	return (struct backend *)b;
}

